#include <avr/io.h>
#include <avr/interrupt.h>
#include <avr/signal.h>
#include <avr/eeprom.h>

#include "lcd_lib.c"

//define dds port
#define DDSPORT	PORTA 
#define DDSDDR 	DDRA

//define HS Port
#define HSDDR	DDRD
#define HSPORT	PORTD
#define HS	5

//define button port and dedicated pins
#define BPORT 	PORTD
#define BPIN 	PIND
#define BDDR 	DDRD
//Müssen zwischen 0 und 4 liegen:
#define BDOWN	0	
#define BLEFT	1	
#define BSTART	2	
#define BRIGHT	3
#define BUP	4


//define Test port
#define TPORT 	PORTB
#define TPIN 	PINB
#define TDDR 	DDRB


//define signals
const uint8_t  sinewave[] PROGMEM= //sine 256 values
{
0x80,0x83,0x86,0x89,0x8c,0x8f,0x92,0x95,0x98,0x9c,0x9f,0xa2,0xa5,0xa8,0xab,0xae,
0xb0,0xb3,0xb6,0xb9,0xbc,0xbf,0xc1,0xc4,0xc7,0xc9,0xcc,0xce,0xd1,0xd3,0xd5,0xd8,
0xda,0xdc,0xde,0xe0,0xe2,0xe4,0xe6,0xe8,0xea,0xec,0xed,0xef,0xf0,0xf2,0xf3,0xf5,
0xf6,0xf7,0xf8,0xf9,0xfa,0xfb,0xfc,0xfc,0xfd,0xfe,0xfe,0xff,0xff,0xff,0xff,0xff,
0xff,0xff,0xff,0xff,0xff,0xff,0xfe,0xfe,0xfd,0xfc,0xfc,0xfb,0xfa,0xf9,0xf8,0xf7,
0xf6,0xf5,0xf3,0xf2,0xf0,0xef,0xed,0xec,0xea,0xe8,0xe6,0xe4,0xe2,0xe0,0xde,0xdc,
0xda,0xd8,0xd5,0xd3,0xd1,0xce,0xcc,0xc9,0xc7,0xc4,0xc1,0xbf,0xbc,0xb9,0xb6,0xb3,
0xb0,0xae,0xab,0xa8,0xa5,0xa2,0x9f,0x9c,0x98,0x95,0x92,0x8f,0x8c,0x89,0x86,0x83,
0x80,0x7c,0x79,0x76,0x73,0x70,0x6d,0x6a,0x67,0x63,0x60,0x5d,0x5a,0x57,0x54,0x51,
0x4f,0x4c,0x49,0x46,0x43,0x40,0x3e,0x3b,0x38,0x36,0x33,0x31,0x2e,0x2c,0x2a,0x27,
0x25,0x23,0x21,0x1f,0x1d,0x1b,0x19,0x17,0x15,0x13,0x12,0x10,0x0f,0x0d,0x0c,0x0a,
0x09,0x08,0x07,0x06,0x05,0x04,0x03,0x03,0x02,0x01,0x01,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x01,0x02,0x03,0x03,0x04,0x05,0x06,0x07,0x08,
0x09,0x0a,0x0c,0x0d,0x0f,0x10,0x12,0x13,0x15,0x17,0x19,0x1b,0x1d,0x1f,0x21,0x23,
0x25,0x27,0x2a,0x2c,0x2e,0x31,0x33,0x36,0x38,0x3b,0x3e,0x40,0x43,0x46,0x49,0x4c,
0x4f,0x51,0x54,0x57,0x5a,0x5d,0x60,0x63,0x67,0x6a,0x6d,0x70,0x73,0x76,0x79,0x7c
};
const uint8_t squarewave[] PROGMEM= //square wave
{
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
};
const uint8_t sawtoothwave[] PROGMEM= //sawtooth wave
{
0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,0x09,0x0a,0x0b,0x0c,0x0d,0x0e,0x0f,
0x10,0x11,0x12,0x13,0x14,0x15,0x16,0x17,0x18,0x19,0x1a,0x1b,0x1c,0x1d,0x1e,0x1f,
0x20,0x21,0x22,0x23,0x24,0x25,0x26,0x27,0x28,0x29,0x2a,0x2b,0x2c,0x2d,0x2e,0x2f,
0x30,0x31,0x32,0x33,0x34,0x35,0x36,0x37,0x38,0x39,0x3a,0x3b,0x3c,0x3d,0x3e,0x3f,
0x40,0x41,0x42,0x43,0x44,0x45,0x46,0x47,0x48,0x49,0x4a,0x4b,0x4c,0x4d,0x4e,0x4f,
0x50,0x51,0x52,0x53,0x54,0x55,0x56,0x57,0x58,0x59,0x5a,0x5b,0x5c,0x5d,0x5e,0x5f,
0x60,0x61,0x62,0x63,0x64,0x65,0x66,0x67,0x68,0x69,0x6a,0x6b,0x6c,0x6d,0x6e,0x6f,
0x70,0x71,0x72,0x73,0x74,0x75,0x76,0x77,0x78,0x79,0x7a,0x7b,0x7c,0x7d,0x7e,0x7f,
0x80,0x81,0x82,0x83,0x84,0x85,0x86,0x87,0x88,0x89,0x8a,0x8b,0x8c,0x8d,0x8e,0x8f,
0x90,0x91,0x92,0x93,0x94,0x95,0x96,0x97,0x98,0x99,0x9a,0x9b,0x9c,0x9d,0x9e,0x9f,
0xa0,0xa1,0xa2,0xa3,0xa4,0xa5,0xa6,0xa7,0xa8,0xa9,0xaa,0xab,0xac,0xad,0xae,0xaf,
0xb0,0xb1,0xb2,0xb3,0xb4,0xb5,0xb6,0xb7,0xb8,0xb9,0xba,0xbb,0xbc,0xbd,0xbe,0xbf,
0xc0,0xc1,0xc2,0xc3,0xc4,0xc5,0xc6,0xc7,0xc8,0xc9,0xca,0xcb,0xcc,0xcd,0xce,0xcf,
0xd0,0xd1,0xd2,0xd3,0xd4,0xd5,0xd6,0xd7,0xd8,0xd9,0xda,0xdb,0xdc,0xdd,0xde,0xdf,
0xe0,0xe1,0xe2,0xe3,0xe4,0xe5,0xe6,0xe7,0xe8,0xe9,0xea,0xeb,0xec,0xed,0xee,0xef,
0xf0,0xf1,0xf2,0xf3,0xf4,0xf5,0xf6,0xf7,0xf8,0xf9,0xfa,0xfb,0xfc,0xfd,0xfe,0xff
};
const uint8_t rewsawtoothwave[] PROGMEM= //reverse sawtooth wave
{
0xff,0xfe,0xfd,0xfc,0xfb,0xfa,0xf9,0xf8,0xf7,0xf6,0xf5,0xf4,0xf3,0xf2,0xf1,0xf0,
0xef,0xee,0xed,0xec,0xeb,0xea,0xe9,0xe8,0xe7,0xe6,0xe5,0xe4,0xe3,0xe2,0xe1,0xe0,
0xdf,0xde,0xdd,0xdc,0xdb,0xda,0xd9,0xd8,0xd7,0xd6,0xd5,0xd4,0xd3,0xd2,0xd1,0xd0,
0xcf,0xce,0xcd,0xcc,0xcb,0xca,0xc9,0xc8,0xc7,0xc6,0xc5,0xc4,0xc3,0xc2,0xc1,0xc0,
0xbf,0xbe,0xbd,0xbc,0xbb,0xba,0xb9,0xb8,0xb7,0xb6,0xb5,0xb4,0xb3,0xb2,0xb1,0xb0,
0xaf,0xae,0xad,0xac,0xab,0xaa,0xa9,0xa8,0xa7,0xa6,0xa5,0xa4,0xa3,0xa2,0xa1,0xa0,
0x9f,0x9e,0x9d,0x9c,0x9b,0x9a,0x99,0x98,0x97,0x96,0x95,0x94,0x93,0x92,0x91,0x90,
0x8f,0x8e,0x8d,0x8c,0x8b,0x8a,0x89,0x88,0x87,0x86,0x85,0x84,0x83,0x82,0x81,0x80,
0x7f,0x7e,0x7d,0x7c,0x7b,0x7a,0x79,0x78,0x77,0x76,0x75,0x74,0x73,0x72,0x71,0x70,
0x6f,0x6e,0x6d,0x6c,0x6b,0x6a,0x69,0x68,0x67,0x66,0x65,0x64,0x63,0x62,0x61,0x60,
0x5f,0x5e,0x5d,0x5c,0x5b,0x5a,0x59,0x58,0x57,0x56,0x55,0x54,0x53,0x52,0x51,0x50,
0x4f,0x4e,0x4d,0x4c,0x4b,0x4a,0x49,0x48,0x47,0x46,0x45,0x44,0x43,0x42,0x41,0x40,
0x3f,0x3e,0x3d,0x3c,0x3b,0x3a,0x39,0x38,0x37,0x36,0x35,0x34,0x33,0x32,0x31,0x30,
0x2f,0x2e,0x2d,0x2c,0x2b,0x2a,0x29,0x28,0x27,0x26,0x25,0x24,0x23,0x22,0x21,0x20,
0x1f,0x1e,0x1d,0x1c,0x1b,0x1a,0x19,0x18,0x17,0x16,0x15,0x14,0x13,0x12,0x11,0x10,
0x0f,0x0e,0x0d,0x0c,0x0b,0x0a,0x09,0x08,0x07,0x06,0x05,0x04,0x03,0x02,0x01,0x00,
};

const uint8_t trianglewave[] PROGMEM= //triangle wave
{
0x00,0x02,0x04,0x06,0x08,0x0a,0x0c,0x0e,0x10,0x12,0x14,0x16,0x18,0x1a,0x1c,0x1e,
0x20,0x22,0x24,0x26,0x28,0x2a,0x2c,0x2e,0x30,0x32,0x34,0x36,0x38,0x3a,0x3c,0x3e,
0x40,0x42,0x44,0x46,0x48,0x4a,0x4c,0x4e,0x50,0x52,0x54,0x56,0x58,0x5a,0x5c,0x5e,
0x60,0x62,0x64,0x66,0x68,0x6a,0x6c,0x6e,0x70,0x72,0x74,0x76,0x78,0x7a,0x7c,0x7e,
0x80,0x82,0x84,0x86,0x88,0x8a,0x8c,0x8e,0x90,0x92,0x94,0x96,0x98,0x9a,0x9c,0x9e,
0xa0,0xa2,0xa4,0xa6,0xa8,0xaa,0xac,0xae,0xb0,0xb2,0xb4,0xb6,0xb8,0xba,0xbc,0xbe,
0xc0,0xc2,0xc4,0xc6,0xc8,0xca,0xcc,0xce,0xd0,0xd2,0xd4,0xd6,0xd8,0xda,0xdc,0xde,
0xe0,0xe2,0xe4,0xe6,0xe8,0xea,0xec,0xee,0xf0,0xf2,0xf4,0xf6,0xf8,0xfa,0xfc,0xfe,
0xff,0xfd,0xfb,0xf9,0xf7,0xf5,0xf3,0xf1,0xef,0xef,0xeb,0xe9,0xe7,0xe5,0xe3,0xe1,
0xdf,0xdd,0xdb,0xd9,0xd7,0xd5,0xd3,0xd1,0xcf,0xcf,0xcb,0xc9,0xc7,0xc5,0xc3,0xc1,
0xbf,0xbd,0xbb,0xb9,0xb7,0xb5,0xb3,0xb1,0xaf,0xaf,0xab,0xa9,0xa7,0xa5,0xa3,0xa1,
0x9f,0x9d,0x9b,0x99,0x97,0x95,0x93,0x91,0x8f,0x8f,0x8b,0x89,0x87,0x85,0x83,0x81,
0x7f,0x7d,0x7b,0x79,0x77,0x75,0x73,0x71,0x6f,0x6f,0x6b,0x69,0x67,0x65,0x63,0x61,
0x5f,0x5d,0x5b,0x59,0x57,0x55,0x53,0x51,0x4f,0x4f,0x4b,0x49,0x47,0x45,0x43,0x41,
0x3f,0x3d,0x3b,0x39,0x37,0x35,0x33,0x31,0x2f,0x2f,0x2b,0x29,0x27,0x25,0x23,0x21,
0x1f,0x1d,0x1b,0x19,0x17,0x15,0x13,0x11,0x0f,0x0f,0x0b,0x09,0x07,0x05,0x03,0x01
};
const uint8_t ECG[] PROGMEM= //ECG wave
{
73,74,75,75,74,73,73,73,73,72,71,69,68,67,67,67,
68,68,67,65,62,61,59,57,56,55,55,54,54,54,55,55,
55,55,55,55,54,53,51,50,49,49,52,61,77,101,132,
169,207,238,255,254,234,198,154,109,68,37,17,5,
0,1,6,13,20,28,36,45,52,57,61,64,65,66,67,68,68,
69,70,71,71,71,71,71,71,71,71,72,72,72,73,73,74,
75,75,76,77,78,79,80,81,82,83,84,86,88,91,93,96,
98,100,102,104,107,109,112,115,118,121,123,125,
126,127,127,127,127,127,126,125,124,121,119,116,
113,109,105,102,98,95,92,89,87,84,81,79,77,76,75,
74,73,72,70,69,68,67,67,67,68,68,68,69,69,69,69,
69,69,69,70,71,72,73,73,74,74,75,75,75,75,75,75,
74,74,73,73,73,73,72,72,72,71,71,71,71,71,71,71,
70,70,70,69,69,69,69,69,70,70,70,69,68,68,67,67,
67,67,66,66,66,65,65,65,65,65,65,65,65,64,64,63,
63,64,64,65,65,65,65,65,65,65,64,64,64,64,64,64,
64,64,65,65,65,66,67,68,69,71,72,73
};



/*define LCD Strucktur:
 *0:
 *   0123456789012345
 * 0|DDS-SIGNAL: AAAA|0
 * 1|XXXXXHz TTTTTTTT|1
 *   0123456789012345
 *1:
 *   0123456789012345
 * 0|HSOUT: BBBB XMHZ|0
 * 1|XXXXXHz TTTTTTTT|1
 *   0123456789012345
 *2:   
 *   0123456789012345
 * 0|SW:yyYYs XXXXXHz|0
 * 1|ZZZZZHz TTTTTTTT|1
 *   0123456789012345
 *3:
 *   0123456789012345
 * 0|Save Config:    |0
 * 1|Press Start     |1
 *   0123456789012345
 *
 * 
 * AAAA=> OFF_ / ON__ / LIVE
 * BBBB=> OFF_ / ON__
 * X=> Frequenz
 * T=> TYP
 * S=> Start Frequenz
 * Z=> Stop Frequenz
 * yy=> Steptyp: LI, LO
 * YY=> Sekunden
 *
 */

uint8_t get_maxSteps(uint8_t step){
	switch(step){
		//Main, Mode, 5xFrequnz, Signal
		case 0:	 return 8;
		//Main, Mode, Hsfrequenz, 5xFrequnz, Signal
		case 1:  return 9;
		//Main, OptionLinLog, 2xZeit, 5xStartfrequnz, 5xStopfrequnz, Signal
		case 2:	 return 15;
		//Main
		case 3:  return 1;
		default: return 0;
	}
}

uint8_t get_Variable_for_Step(uint8_t step, uint8_t i){
	switch(step){
		case 0:
			return i;
		case 1:
			switch(i){
				case 0:  return 0;
				case 1:  return 8;
				case 2:  return 22;
				default: return 2+i-3;
			}
		case 2:
			switch(i){
				case 0:  return 0;
				case 14: return 7;
				default:  return 9 + i-1;
			}
		case 3:
			return 0;
		default:
			return -1;
	}
}

int8_t variablen[]={
	0, //00: akt Mode	
	2, //01: AAAA -0:OFF/ -1:ON/ -2:LIVE	
	0, //02: Frequenz 10^4 (X)	
	3, //03: Frequenz 10^3 (X)	
	0, //04: Frequenz 10^2 (X)	
	0, //05: Frequenz 10^1 (X)	
	5, //06: Frequenz 10^0 (X)
        0, //07: Signal (TTTTTT)
	1, //08: HS ON/OF	
	0, //09: Steptyp (yy)
	0, //10: Sekunden 10^1 (Y)
	0, //11: Sekunden 10^0 (Y)
	0, //12: StartFrequenz 10^4 (S)	
	0, //13: StartFrequenz 10^3 (S)	
	0, //14: StartFrequenz 10^2 (S)	
	0, //15: StartFrequenz 10^1 (S)	
	0, //16: StartFrequenz 10^0 (S)
	0, //17: StopFrequenz 10^4 (Z)	
	0, //18: StopFrequenz 10^3 (Z)	
	0, //19: StopFrequenz 10^2 (Z)	
	0, //20: StopFrequenz 10^1 (Z)	
	0, //21: StopFrequenz 10^0 (Z)
	0, //22: HS Frequenz 1, 2, 4, 8 
};

enum variablen_Parameter {MAX=1, LCDPOSX, LCDPOSY };

int8_t get_Variablen_Parameter(uint8_t variable, enum variablen_Parameter p){
  switch(variable){
	  case 0: //akt Mode 
	  	switch(p){
			case MAX:	return 3;
			case LCDPOSX:	return 0;
			case LCDPOSY:	return 0;
			default:	return -1;
		}
	  case 1: //AAAA  OFF ON LIVE 
	  	switch(p){
			case MAX: 	return 2;
			case LCDPOSX:	return 12;
			case LCDPOSY:	return 0;
			default:	return -1;
		}
	  case 2: //Frequnz 10^4 => 10^0
	  case 3:
	  case 4:
	  case 5:
	  case 6: 
	  	switch(p){
			case MAX:	return 9;
			case LCDPOSX:	return 0 +variable-2;
			case LCDPOSY:	return 1;
			default:	return -1;
		}
	  case 7: //Signal 
	  	switch(p){
			case MAX:	return 5;
			case LCDPOSX:	return 8;
			case LCDPOSY:	return 1;
			default:	return -1;
		}
	  case 8: //HS ON OF 
	  	switch(p){
			case MAX:	return 1;
			case LCDPOSX:	return 7;
			case LCDPOSY:	return 0;
			default:	return -1;
		}
	  case 9: // StepTyp 
	  	switch(p){
			case MAX:	return 1;
			case LCDPOSX:	return 3;
			case LCDPOSY:	return 0;
			default:	return -1;
		}
	  case 10: //Sekunden
	  case 11: 
	  	switch(p){
			case MAX:	return 9;
			case LCDPOSX:	return 5 + variable-10;
			case LCDPOSY:	return 0;
			default:	return -1;
		}
	  case 12:  //Start Frequenz
	  case 13:
	  case 14:
	  case 15:
	  case 16: 
	  	switch(p){
			case MAX:	return 9;
			case LCDPOSX:	return 9 + variable-12;
			case LCDPOSY:	return 0;
			default:	return -1;
		}
	  case 17: //Stop Frequenz
	  case 18:
	  case 19:
	  case 20:
	  case 21:
	  	switch(p){
			case MAX:	return 9;
			case LCDPOSX:	return 0 + variable-17;
			case LCDPOSY:	return 1;
			default:	return -1;
		}
	  case 22: 
	  	switch(p){
			case MAX:	return 3;
			case LCDPOSX:	return 12;
			case LCDPOSY:	return 0;
			default:	return -1;
		}
	  default:
	  	return -1;
  }

}


//Define LCD Strings
const prog_char LCDStringMain00[] ="DDS-SIGNAL:     ";
const prog_char LCDStringMain01[] ="HSOUT:       MHZ";
const prog_char LCDStringMain02[] ="SW:    s      Hz";
const prog_char LCDStringMain03[] ="Save Config:    ";
const prog_char LCDStringMain1012[] ="     Hz         ";
const prog_char LCDStringMain13[] ="Press Start     ";
const prog_char LCDStringSignal00[] ="Sinus   ";
const prog_char LCDStringSignal01[] ="Rechteck";
const prog_char LCDStringSignal02[] ="S\x84gezahn";
const prog_char LCDStringSignal03[] ="inv S\x84g.";
const prog_char LCDStringSignal04[] ="Dreieck ";
const prog_char LCDStringSignal05[] ="ECG     ";
const prog_char LCDStringHS0[] ="1";
const prog_char LCDStringHS1[] ="2";
const prog_char LCDStringHS2[] ="4";
const prog_char LCDStringHS3[] ="8";
const prog_char LCDStringStat0[] ="OFF ";
const prog_char LCDStringStat1[] ="ON  ";
const prog_char LCDStringStat2[] ="LIVE";
const prog_char LCDStringSWTYP0[] ="LI";
const prog_char LCDStringSWTYP1[] ="LO";
const prog_char LCDStringZ0[] ="0";
const prog_char LCDStringZ1[] ="1";
const prog_char LCDStringZ2[] ="2";
const prog_char LCDStringZ3[] ="3";
const prog_char LCDStringZ4[] ="4";
const prog_char LCDStringZ5[] ="5";
const prog_char LCDStringZ6[] ="6";
const prog_char LCDStringZ7[] ="7";
const prog_char LCDStringZ8[] ="8";
const prog_char LCDStringZ9[] ="9";
const prog_char LCDStringERROR[] ="X";

const prog_char* get_Variable_String(uint8_t variable){
	switch(variable){
		case 0: //Menue muss selbst verwaltet werden
			return LCDStringERROR;
		case 1: //OFF ON LIVE:
		case 8:
			switch(variablen[variable]){
				case 0:		return LCDStringStat0;	
				case 1:		return LCDStringStat1;
				case 2:		return LCDStringStat2;
				default:	return LCDStringERROR;
			}		
		case 2: //Zahlen
		case 3:
		case 4:
		case 5:
		case 6:
		case 10:
		case 11:
		case 12:
		case 13:
		case 14:
		case 15:
		case 16:
		case 17:
		case 18:
		case 19:
		case 20:
		case 21:
			switch(variablen[variable]){
				case 0:		return LCDStringZ0;
				case 1:		return LCDStringZ1;
				case 2:		return LCDStringZ2;
				case 3:		return LCDStringZ3;
				case 4:		return LCDStringZ4;
				case 5:		return LCDStringZ5;
				case 6:		return LCDStringZ6;
				case 7:		return LCDStringZ7;
				case 8:		return LCDStringZ8;
				case 9:		return LCDStringZ9;
				default:	return LCDStringERROR;
			}
		case 7: //Signal
			switch(variablen[variable]){
				case 0:		return LCDStringSignal00; 
				case 1:		return LCDStringSignal01; 
				case 2:		return LCDStringSignal02; 
				case 3:		return LCDStringSignal03; 
				case 4:		return LCDStringSignal04; 
				case 5:		return LCDStringSignal05; 
				default:	return LCDStringERROR;
			}
		case 9: //HS LI LO
			switch(variablen[variable]){
				case 0:		return LCDStringSWTYP0;
				case 1:		return LCDStringSWTYP1;
				default:	return LCDStringERROR;
			}
		case 22: //HS Frequenz
			switch(variablen[variable]){
				case 0:		return LCDStringHS0;
				case 1:		return LCDStringHS1;
				case 2:		return LCDStringHS2;
				case 3:		return LCDStringHS3;
				default:	return LCDStringERROR;
			}
		default:	
			return LCDStringERROR;
	}
}


const prog_char* get_MAIN_String(uint8_t mode, uint8_t zeile){
	switch(mode){
		case 0:
			if (zeile==0) 	return LCDStringMain00;
			else		return LCDStringMain1012;
		case 1:
			if (zeile==0) 	return LCDStringMain01;
			else		return LCDStringMain1012;
		case 2:
			if (zeile==0) 	return LCDStringMain02;
			else		return LCDStringMain1012;
		case 3:
			if (zeile==0) 	return LCDStringMain03;
			else		return LCDStringMain13;
	}
	return LCDStringERROR;
}

void saveData(void){
	uint8_t temp;
	//Codewort
	eeprom_write_byte((uint8_t*)0,'D');
	for (temp=0; temp < sizeof(variablen);temp++)
		eeprom_write_byte((uint8_t*)(temp +1),variablen[temp]);
}



//define global variables
int8_t BlastTast=-1;
unsigned Bstable=1;
int8_t aktStep=0;
int8_t aktMode=0;
int8_t aktSignal_i=-1;
uint8_t aktSignal[256] __attribute__ ((section (".noinit"))); 
uint8_t ad2=0x00;
uint8_t ad1=0x00;
uint8_t ad0=0xFF;


//load signal into ram:
void load_Signal(int8_t signal){
	const uint8_t* psignal;
	
	if (aktSignal_i==signal) return;

	switch(signal){
		case 0:  psignal=sinewave; break;
		case 1:  psignal=squarewave; break;
		case 2:  psignal=sawtoothwave; break;
		case 3:  psignal=rewsawtoothwave; break;
		case 4:  psignal=trianglewave; break;
		case 5:  psignal=ECG; break;
		default: psignal=sinewave; break;
	}
	
	uint16_t i;	
	for (i=0;i<sizeof(aktSignal);i++)
 		aktSignal[i]=pgm_read_byte(psignal+i);

	aktSignal_i=signal;	
}

void init_all(){
  uint8_t temp;

  //Init DDS
  DDSDDR=0xFF;//set DDS port as output
  DDSPORT=0x00;//set initial zero values

  //Init Taster Port
//Port comp heck
//  BDDR=0x02;//set port as input
 BDDR=0x00; 
 BPORT=0xFF;//set pullups

  //Init entprellungstimer
  TCCR0&=~((1<<WGM00)|(1<<WGM01)|(1<<CS01)|(1<<CS00)|(1<<CS02));
  TIMSK|=(1<<TOIE0);

  //Init HS output
  HSPORT&=~(1<<HS);	//Set output to low
  HSDDR|=(1<<HS);	//configure as output

  //Init Testport
  TDDR=0xFF;
  TPORT=0x00;//set initial zero values

  //Werte Lesen
  if (eeprom_read_byte((uint8_t*)0)!='D'){
  	saveData();	
  }
  for (temp=0; temp < sizeof(variablen);temp++)
	variablen[temp]=eeprom_read_byte((uint8_t*)(temp +1));
  
  //Init LCD
  LCDinit();
  LCDclr();
  LCDcursorOnBlink();
  
  //Interrups erlauben
  sei();

}

//Wird ~5ms nachdem eine Taste gedrückt wurde aufgerufen
SIGNAL(SIG_OVERFLOW0){
  //Timer deaktivieren:
  TCCR0&=~(1<<CS02);
  //Taster wert als Stabiel bezeichnen
  Bstable=1;
}

int8_t get_Tast(){
  uint8_t temp;
  int8_t aktTast;
  //Aktuelle Teste Bestimmen:
  temp= (~BPIN) & ((1<<BDOWN)|(1<<BLEFT)|(1<<BSTART)|(1<<BRIGHT)|(1<<BUP));
  switch(temp){
    case (1<<BDOWN):
      aktTast=BDOWN;
    break;
    case (1<<BLEFT):
      aktTast=BLEFT;
    break;
    case (1<<BSTART):
      aktTast=BSTART;      
    break;
    case (1<<BRIGHT):
      aktTast=BRIGHT;            
    break;
    case (1<<BUP):
      aktTast=BUP;      
    break;
    default:
      aktTast=-1;
  }
 
  if (Bstable==1){
    //Taster wert hat sich nicht geändert!
    if (BlastTast==aktTast){
      return -1;
    }
    //Tasten wert hat sich geändert:
    
    //Timer aktivieren:
    Bstable=0;    
    TCCR0|=(1<<CS02);

    //Neuen Tasten wert übernehmen
    BlastTast=aktTast;
    return aktTast;
  }
  
  return -1;
}


void meneuNeuAufbauen(void){
	uint8_t i;	
	uint8_t variable;
	//Main Strings ausgeben:
	CopyStringtoLCD(get_MAIN_String(variablen[0],0),0,0);
	CopyStringtoLCD(get_MAIN_String(variablen[0],1),0,1);
  	//Werte Laden:
	for(i=1; i<get_maxSteps(variablen[0]);i++){
		variable=get_Variable_for_Step(variablen[0],i);
		CopyStringtoLCD(
			get_Variable_String(variable),
			get_Variablen_Parameter(variable,LCDPOSX),
			get_Variablen_Parameter(variable,LCDPOSY)
		);
	}
	//Courser setzen
	variable=get_Variable_for_Step(variablen[0],aktStep);
	LCDGotoXY(
		get_Variablen_Parameter(variable,LCDPOSX),
		get_Variablen_Parameter(variable,LCDPOSY)
	);
}

void calcDDS(void){
	uint32_t temp;
	//Je nach mode:
	switch(variablen[0]){
		case 0:	//Normal + HS gehört zusammen:
		case 1:
		default://01: AAAA -0:OFF/ -1:ON/ -2:LIVE
			if (variablen[1]!=0){			
				//Frequnz ausrechnen:
				temp=	(uint32_t)variablen[6]+
					(uint32_t)variablen[5]*10 +
					(uint32_t)variablen[4]*100 +
					(uint32_t)variablen[3]*1000 +
					(uint32_t)variablen[2]*10000;
				temp=temp/0.095367431640625;
				ad2=(uint8_t)((uint32_t)temp>>16);
				ad1=(uint8_t)((uint32_t)temp>>8);
				ad0=(uint8_t)temp;
			}
			else{	//Signal aus => GND
				ad2=0;
				ad1=0;
				ad0=0;
			}
			if (variablen[8]!=0){ //HS ON
				switch(variablen[22]){	//22: HS Frequenz 1, 2, 4, 8
					case 0: OCR1A=7; break; //start high speed (1MHz) signal
					case 1: OCR1A=3; break; //2MHz
					case 2: OCR1A=1; break; //4MHz
					case 3: OCR1A=0; break; //8MHz
					default: OCR1A=7; //1MHz
				};
				
				//Output compare toggles OC1A pin
				TCCR1A=0x40;
				//start timer without prescaler
				TCCR1B=0x09;//0b00001001;
			}
			else //HS OFF
				TCCR1B=0x00;//timer off
						
		break;
		case 2: //Sweep 
		break;
		case 3: //SaveData
			variablen[0]=aktMode;
			saveData();
			meneuNeuAufbauen();
			calcDDS();
		break;
	}
	//Signalform laden:
	load_Signal(variablen[7]);

	aktMode=variablen[0];
}


int main () {
	uint8_t variable;
	uint8_t altBPIN;

	init_all();
	meneuNeuAufbauen();
	
	calcDDS();

	//Akt Taste abfragen 
	altBPIN=BPIN;		

	for(;;)
	{
		int8_t aktTaste = get_Tast();
		//Step änder
		if (aktTaste==BLEFT || aktTaste==BRIGHT){
			if (aktTaste==BLEFT)
				aktStep--;
			else
				aktStep++;
			
			if(aktStep<0)
				aktStep=get_maxSteps(variablen[0])-1;			
			else if(aktStep>=get_maxSteps(variablen[0]))
				aktStep=0;

			//Courser neu setzen
			variable=get_Variable_for_Step(variablen[0],aktStep);
			LCDGotoXY(
				get_Variablen_Parameter(variable,LCDPOSX),
				get_Variablen_Parameter(variable,LCDPOSY)
			);			
		}
		//value ändern
		if (aktTaste==BUP || aktTaste==BDOWN){
			variable=get_Variable_for_Step(variablen[0],aktStep);
			if (variable==0){
				if(aktTaste==BUP)
					variablen[variable]--;				
				else
					variablen[variable]++;
			}
			else{			
				if(aktTaste==BUP)
					variablen[variable]++;				
				else
					variablen[variable]--;
			}			
			if(variablen[variable]<0)	
				variablen[variable]=get_Variablen_Parameter(variable,MAX);
			if(variablen[variable]>get_Variablen_Parameter(variable,MAX))
				variablen[variable]=0;
			
			if(variable!=0){
				//Wert neu ausgeben
				CopyStringtoLCD(
					get_Variable_String(variable),
					get_Variablen_Parameter(variable,LCDPOSX),
					get_Variablen_Parameter(variable,LCDPOSY)
				);
				LCDGotoXY(
					get_Variablen_Parameter(variable,LCDPOSX),
					get_Variablen_Parameter(variable,LCDPOSY)
				);			
				
				//Wenn Live dann änderung sofort übernehmen
				if(variablen[1]==2 && (variablen[0]==0 || variablen[0]==1))
					calcDDS();
			}
			//Modus wechsel:
			else
				meneuNeuAufbauen();
		}

		//DDS neu einstellen
		if (aktTaste==BSTART){
			calcDDS();
		}

		//DDS Starten oder nur 0 ausgeben 
		if (ad0!=0 || ad1!=0 || ad2!=0){
			//DDS algorithmus
			altBPIN &= ~(1<<HS);
			if ((altBPIN & 0x1F)==0x1F){//warte auf Taster gedrückt
				asm volatile(	
					"eor r18, r18 		;r18<-0"		"\n\t"
					"eor r19, r19 		;r19<-0"		"\n\t"
					"DDSloop1:"					"\n\t"
					"add r18, %0		;1 cycle"		"\n\t"
					"adc r19, %1		;1 cycle"		"\n\t"
					"adc %A3, %2		;1 cycle"		"\n\t"
					"ld __tmp_reg__, Z	;2 cycles" 		"\n\t"
					"out %4, __tmp_reg__	;1 cycle"		"\n\t"
					"in r20, %5		;1 cycle"		"\n\t"
					"cp r20, %6 		;1 cycle"		"\n\t"
					"brsh DDSloop1		;2 cycles=>10 cycles"	"\n\t"
					"mov %6, r20		;altBPin<-r20"		"\n\t"
					:
					:"r" (ad0),"r" (ad1),"r" (ad2),
						"e" (aktSignal),"I" (_SFR_IO_ADDR(DDSPORT)), 
						"I" (_SFR_IO_ADDR(BPIN)), "r" (altBPIN)
					:"r18", "r19", "r20"
	    			);
			}
			else{ //Warte auf Taster losgelassen:
				altBPIN= ~altBPIN & 0x1F;
				asm volatile(	
					"eor r18, r18 		;r18<-0"		"\n\t"
					"eor r19, r19 		;r19<-0"		"\n\t"
					"DDSloop2:"					"\n\t"
					"add r18, %0		;1 cycle"		"\n\t"
					"adc r19, %1		;1 cycle"		"\n\t"
					"adc %A3, %2		;1 cycle"		"\n\t"
					"ld __tmp_reg__, Z	;2 cycles" 		"\n\t"
					"out %4, __tmp_reg__	;1 cycle"		"\n\t"
					"in r20, %5		;1 cycle"		"\n\t"
					"and r20, %6 		;1 cycle"		"\n\t"
					"breq DDSloop2		;2 cycles=>10 cycles"	"\n\t"
					"in %6, %5		;altBPIN<-BPIN"		"\n\t"
					:
					:"r" (ad0),"r" (ad1),"r" (ad2),
						"e" (aktSignal),"I" (_SFR_IO_ADDR(DDSPORT)), 
						"I" (_SFR_IO_ADDR(BPIN)), "r" (altBPIN)
					:"r18", "r19", "r20"
	    			);
			}
		}
		else{//Nullliene wenn DDS inaktiv
			uint8_t tempPin=BPIN;
			DDSPORT=128;
			while(altBPIN==tempPin)
				tempPin=BPIN;
			altBPIN=tempPin;
		}
  
	}	
}


